#!/usr/bin/env python

#   trek.bas   27-Jul-73   Aron K. Insinga   Project Delta
#
#   http://pdp-11.trailing-edge.com/rsts11/rsts-11-020/
#
# Converted to python 26 Aug 2024, Mike Markowski, mike.ab3ap@gmail.com
#
# Conversion included introduction of subroutines and converting arrays to
# start at 0 rather 1 and math changes that that required.  I also converted
# output text from all upper to mixed case and even discovered a bug (noted
# in warp().

from math import atan2, cos, pi, sin, sqrt
from urandom import randint, random
from tml import tml

ds = ['Warp Engines', 'S.R. Sensors', 'L.R. Sensors',
    'Phaser Cntrl', 'Photon Tubes', 'Damage Cntrl', 'Computer']
d = [0,0,0,0,0,0,0] # Damage control report.
# Galaxy, 8x8 quadrants.
g = [[0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0]]
# Map of explored quadrants, 8x8 quadrants.
m = [[0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0]]
# Quadrant, 8x8 sectors.
q = [[0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0],
     [0,0,0,0,0,0,0,0]]
# Klingons, sector location and energy level.
k = [[0,0,0],
     [0,0,0],
     [0,0,0],
     [0,0,0],
     [0,0,0],
     [0,0,0],
     [0,0,0],
     [0,0,0],
     [0,0,0]]

t = t0 = 100*randint(20, 39) # Current and start Stardate.
t9 = 40          # Solar years to complete mission.
e = e0 = 3000    # Current and initial Enterprise energy level.
p = p0 = 10      # Current and initial number of photon torpedos.
a = 0            # 0 to allow Klingon attack.
b9 = k3 = k9 = 0 # Number of Klingons in quadrant and galaxy.
q1 = q2 = 0      # Current quadrant.
s1 = s2 = 0      # Current sector.
s9 = 200         # Klingon initial energy level.
c = 'Green'

tr = tml(dark_mode = True, tab_size = 8)

def adjCells(a, rLoc, cLoc):
    r0 = max(0, rLoc-1)
    r1 = min(7, rLoc+1) + 1
    c0 = max(0, cLoc-1)
    c1 = min(7, cLoc+1) + 1
    return a[r0:r1, c0:c1]

def chartPrint():
    tr.print('\n    ', end='')
    for i in range(8):
        tr.print('%d   ' % (i+1), end='')
    tr.print('\n  ' + 33*'-')
    for i in range(8):
        tr.print('%d ' % (i+1), end='')
        tr.print(8*':   ' + ':')
        tr.print('  ' + 33*'-')

def computer():
    global b9,k,k9,d,ds,q1,q2,s1,s2,t,t0,t9
    if d[6] < 0:
        tr.print('Computer is disabled')
        return
    while True:
        a = tr.input('Computer active and awaiting command: ', alpha=False)
        a = a.strip()
        if a != '':
            a = a[0]
        if a in ['0', '1', '2', '3']:
            break
        tr.print('Functions available from computer')
        tr.print('   0 = Cumulative galactic record')
        tr.print('   1 = Status report')
        tr.print('   2 = Photon torpedo data')
        tr.print('   3 = Direction calculator')
    if a == '0':
        mapPrint(m, q1, q2)
    elif a == '1':
        tr.print('\n   Status Report\n')
        tr.print('Number of Klingons left = %d' % k9)
        tr.print('Number of Stardates left = %d' % (t0+t9-t))
        tr.print('Number of Starbases left = %d' % b9)
        damageRpt()
    elif a == '2':
        for i in range(9):
            if k[i][2] <= 0:
                continue
            rng, theta = vec(s2, s1, k[i][1], k[i][0])
            tr.print(' Range %.1f, bearing %.1f' % (rng, theta))
    else:
        tr.print('You are at quadrant (%d,%d) sector (%d,%d)'
            % (q2+1, q1+1, s2+1, s1+1))
        line = tr.input('Src/dest coords: ', alpha=False)
        try:
            line = line.replace(',', ' ').split()
            line = map(int, line)
            y0, x0, y1, x1 = line
        except ValueError:
            return
        rng, theta = vec(y0, x0, y1, x1)
        tr.print(' Range %.1f, bearing %.1f (sector warp %.2f)'
            % (rng, theta, 0.125*rng))

def damageRpt():
    # Damage control report
    global d,ds
    if d[5] < 0:
        return
    tr.print('\n%s\t\t%s\n' % ('Device','State'))
    for i in range(7):
        tr.print('%s\t%5d' % (ds[i], d[i]))

def emptySector():
    # Find an empty sector in this quadrant.
    global q
    while True:
        r1 = randint(0,7)
        r2 = randint(0,7)
        if q[r1][r2] == 0:
            return r1,r2

def galaxySetup():
    global g,k0,k3,k9,q1,q2,s1,s2

    r = [0.0001, 0.01, 0.03, 0.08, 0.28, 1.28, 3.28, 6.28, 13.28]
    q1 = randint(0,7)
    q2 = randint(0,7)
    s1 = randint(0,7)
    s2 = randint(0,7)
    #   Set up galaxy
    b9 = k9 = 0
    for i in range(8):
        for j in range(8):
            k3 = 0
            c1 = random()*64
            for k0 in range(9): # Max of 9 Klingons per quadrant.
                if c1 < r[k0]:
                    k3 += 1     # Count Klingons in quadrant.
            k9 += k3            # Count Klingons in galaxy.
            if random() > 0.90: # Star base.
                b3 = 1
            else:
                b3 = 0
            b9 += b3            # Count star bases in galaxy.
            g[i][j] = k3*100 + b3*10 + randint(1,8) # Klingon/Base/Stars
    k0 = k9                     # Original number of Klingons.
    if b9 == 0:                 # Ensure at least 1 star base exists.
        i = randint(0,7)
        j = randint(0,7)
        g[i][j] += 10
        b9 = 1

def inputSafe(dtype, prompt):
    '''Keep trying for correct input until user does it right!  This prevents
    the game from dying on incorrect type of response, e.g., accidentally
    entering a text command when a number is expected.
    '''

    while True:
        try:
            ans = dtype(tr.input(prompt, alpha=False))
            return ans
        except ValueError:
            tr.print(ans)
            pass

def klingonAttack():
    # Klingon attack
    global e,k,k3,s1,s2

    if k3 == 0:      # No Klingons in this quadrant.
        return False # Game not over.
    if c == 'Docked':
        tr.print('Starbase shields protect Enterprise')
        return False
    tr.print('')
    for i in range(9):
        if k[i][2] > 0: # Look for Klingon.
            # Hit is Klingon health reduced by distance.
            dist = sqrt((k[i][0] - s1)**2 + (k[i][1] - s2)**2)
            h = int((k[i][2]/dist)*(2 + random())) + 1
            e = max(0, e-h) # Reduce Enterpise health.
            tr.print('%s unit hit from Klingon at sector %d-%d (%d units left)'
                % (h, 1+k[i][1], 1+k[i][0] ,e))
            if e == 0:
                return lose() # Enterprise destroyed.
    return False # Game not over.

def longRange():
    # Long range sensor scan
    global d,g,m,q1,q2

    if d[2] < 0:
        tr.print('Long range sensors are inoperable')
        return
    tr.print('Long range sensor scan for quadrant %d-%d\n' % ((q2+1), (q1+1)))
    for i in range(q1-1, q1+2):
        for j in range(q2-1, q2+2):
            tr.print('  ', end='')             # Indent output.
            if (0 <= i <= 7) and (0 <= j <= 7): # Quadrant is in galaxy.
                tr.print('%03d' % g[i][j], end='')
                m[i][j] = g[i][j]               # Remember explored regions.
            else:                               # Quadrant is outside galaxy.
                tr.print('000', end='')
        tr.print('')

def lose():
    # You lose.
    global k9,t

    tr.print('\tIt is Stardate %d\n' % t)
    tr.print('The Enterprise has been destroyed.')
    tr.print('The Federation will be conquered.')
    tr.print('There are still %d Klingon battle cruisers.' % k9)
    tr.print('You are dead.')
    return True # Game over. :-(

def mapPrint(myMap, row=-1, col=-1):

    tr.print('\n    ', end='')
    for i in range(8):
        tr.print('%d   ' % (i+1), end='') # Print column numbers.
    tr.print('\n  ' + 33*'-')             # Dashed lines.
    for i in range(8):
        tr.print('%d ' % (i+1), end='')   # Print row number.
        for j in range(8):
            sep = '>' if i==row and j==col else '|'
            info = ' - ' if myMap[i][j]==0 else '%03d'%myMap[i][j]
            tr.print('%s%s' % (sep, info), end='')
        tr.print('|')                     # Close row.
    tr.print('  ' + 33*'-')               # Dashed lines.

def phasers():
    # Phaser control
    global d,e,g,k,k3,k9,m,q1,q2,s1,s2

    if d[3] < 0:
        tr.print('Phaser control is disabled')
        return False # Game not over.
    # tr.print('Phasers locked in on target.  Energy available  = %d' % e)
    tr.print('Phasers locked on target.  Energy available = %d' % e)
    while True:
        x = inputSafe(float, 'Number of units to fire? ')
        if x <= 0: # Never mind, leave.
            return False # Game not over.
        elif x > e: # Can't use more than you have.  Try again.
            continue
        else:
            break
    e -= x # Reduce Enterprise health.
    for i in range(9): # Fire on all Klingons in quadrant.
        if k[i][2] <= 0:
            continue
        dist = sqrt((k[i][0] - s1)**2 + (k[i][1] - s2)**2)
        h = int(x/dist*(2 + random()))
        k[i][2] -= h # Reduce Klingon health.
        tr.print('%d unit hit on Klingon at sector %d-%d'
            % (h, 1+k[i][1], 1+k[i][0]))
        if k[i][2] > 0:
            # tr.print('   (%d left)' % k[i][2])
            tr.print('   (%d units left)' % k[i][2])
            continue
        # tr.print('Klingon at sector %d-%d destroyed!' % (1+k[i][1],
        # 1+k[i][0]))
        tr.print('   Klingon destroyed!')
        k3 -= 1                 # One less Klingon in quadrant.
        k9 -= 1                 # One less Klingon in galaxy.
        q[k[i][0]][k[i][1]] = 0 # Remove Klingon from quadrant.
        g[q1][q2] -= 100        # Remove Klingon from galaxy.
        m[q1][q2] -= 100        # Remove Klingon from explored map.
        if k9 == 0:             # All Klingons destroyed!
            return win()        # Game over, we win.
    return False                # Game not over.

def quadrantSetup():
    # Set up quadrant

    global a,b3,k,k3,q,q1,q2,s1,s2,s3,s9

    #tr.print('You are currently in quadrant %d-%d ' % ((q2+1), (q1+1)))
    # Verify we're in the galaxy.
    if not ((0 <= q1 <= 7) and (0 <= q2 <= 7)):
        tr.print('Quadrant %d-%d not in galaxy!' % ((q2+1), (q1+1)))
        return
    k3 = b3 = s3 = 0
    # Clear old Klingons.
    k = [[0,0,0],
         [0,0,0],
         [0,0,0],
         [0,0,0],
         [0,0,0],
         [0,0,0],
         [0,0,0],
         [0,0,0],
         [0,0,0]]
    x = g[q1][q2]          # Quadrant info from galaxy.
    k3 = int(x/100)        # Klingons, left digit.
    b3 = int(x/10) - 10*k3 # Bases, middle digit.
    s3 = x - int(x/10)*10  # Stars, right digit.

    # Clear old quadrant.
    q = [[0,0,0,0,0,0,0,0],
         [0,0,0,0,0,0,0,0],
         [0,0,0,0,0,0,0,0],
         [0,0,0,0,0,0,0,0],
         [0,0,0,0,0,0,0,0],
         [0,0,0,0,0,0,0,0],
         [0,0,0,0,0,0,0,0],
         [0,0,0,0,0,0,0,0]]
    q[s1][s2] = 1          # Place Enterprise in its sector.
    for i in range(k3):    # Place Klingons randomly.
        r1,r2 = emptySector()
        q[r1][r2] = 2
        k[i][0] = r1
        k[i][1] = r2
        k[i][2] = s9       # Initial energy level.
    for i in range(b3):    # Place star bases randomly.
        r1,r2 = emptySector()
        q[r1][r2] = 3
    for i in range(s3):    # Place stars randomly.
        r1,r2 = emptySector()
        q[r1][r2] = 4
    a = 0                  # Allow Klingon attack.

def shortRange():
    # Short range sensor scan
    global a,c,d,e,e0,g,k3,k9,p,p0,q,q1,q2

    if e <= e0/10:
        c = 'Yellow'
    else:
        c = 'Green'
        for i in range(q1-1, q1+2):
            for j in range(q2-1, q2+2):
                if (0 <= i <= 7) and (0 <= j <= 7):
                    if g[i][j] > 99: # Klingon in adjacent quadrant.
                        c = 'Yellow'
                        i = j = 10 # Force end of loops.

    for i in range(s1-1, s1+2):
        for j in range(s2-1, s2+2):
            if not (0 <= i <= 7) or not (0 <= j <= 7):
                continue
            if q[i][j] == 3:
                c = 'Docked'
                e = e0 # Refresh energy to full.
                p = p0 # Refresh photon torpedos to full.
                i = j = 10 # Break from both loops.
    if c != 'Docked':
        if k3 != 0:
            c = 'Red'
        if a == 0: # Set to 0 only in quadrantSetup().
            if klingonAttack():
                return True # Game over, we lost.
    if d[1] != 0:
        tr.print('Short range sensors are inoperable')
    else:
        m[q1][q2] = g[q1][q2] # Remember this quadrant.
        tr.print('\n ---------------')
        for i in range(8):
            for j in range(8):
                if q[i][j] == 0:
                    tr.print(' .', end='')
                else:
                    tr.print(' ' + '.EKB*'[q[i][j]], end='')
            if   i == 0: tr.print('   Stardate: %d' % t)
            elif i == 1: tr.print('   Condition: %s' % c)
            elif i == 2: tr.print('   Quadrant: %d-%d' % ((q2+1), (q1+1)))
            elif i == 3: tr.print('   Sector: %d-%d' % ((s2+1), (s1+1)))
            elif i == 4: tr.print('   Energy: %d' % e)
            elif i == 5: tr.print('   Photon Torpedos: %d' % p)
            elif i == 6: tr.print('   Klingons: %d' % k9)
            elif i == 7: tr.print('')
        tr.print(' ---------------')
    return False # Game not over.

def torpedos():
    # Photon torpedos
    global b3,d,k3,k9,p,q,q1,q2,s1,s2,s3

    if d[4] < 0:
        tr.print('Photon tubes are not operational')
        return False # Game not over.
    if p == 0:
        tr.print('All photon torpedos expended')
        return False # Game not over.
    while True:
        c1 = inputSafe(float, 'Torpedo course (1-8.99999)? ')
        if c1 == 0:
            return False # Game not over.
        elif not (1 <= c1 < 9):
            continue
        else:
            break
    x1 = -sin((c1 - 1)*pi/4)
    x2 = cos((c1 - 1)*pi/4) # Unit vector, course c1 is multiple of 45 deg.
    x,y = s1,s2             # Start at current sector.
    p -= 1                  # Use 1 photon torpedo.
    tr.print('\nTorpedo track:')
    while True:
        x += x1
        y += x2             # Move along unit vector.
        z1 = int(x + 0.5)
        z2 = int(y + 0.5)   # Nearest sector.
        if not (0 <= z1 <= 7) or not (0 <= z2 <= 7): # Torpedo left quadrant.
            tr.print('Missed')
            return klingonAttack()
        tr.print(' %3.1f-%3.1f' % (y, x))
        if q[z1][z2] != 0:   # Torpedo hit something.
            break
    # Find out what torpedo hit.
    if q[z1][z2] == 2:       # Hit a Klingon.
        tr.print('*** Klingon destroyed ***')
        for i in range(9):
            if (k[i][0],k[i][1]) == (z1,z2):
                k[i][2] = 0 # Remove destroyed Klingon.
        k3 -= 1             # Klingon count in quadrant.
        k9 -= 1             # Klingon count in galaxy.
        if k9 == 0:
            return win()    # Game over, we won!
    elif q[z1][z2] == 4:    # Hit a star.
        tr.print('Star destroyed')
        s3 -= 1             # Star count in quadrant.
    else:                   # Hit a star base!
        tr.print('*** Starbase destroyed... Congratulations ***')
        b3 = 0              # Star base count in quadrant.
    q[z1][z2] = 0           # Remove destroyed object.
    g[q1][q2] = k3*100 + b3*10 + s3 # Update this quadrant of galaxy.
    m[q1][q2] = k3*100 + b3*10 + s3 # Update explored.
    return klingonAttack()

def trek():
    global a

    over = False # Game is not over.
    galaxySetup()
    welcome()
    quadrantSetup()
    shortRange()
    while not over: # Let's play a game!
        a = 1 # Disallow Klingon attack.
        ans = tr.input('\nCommand? ', alpha=True).strip()
        if ans == '':
            continue
        ans = ans[0].lower() # 1st char to lower case.
        if ans == 'c': # Course set.
            over = warp() # a=0 to allow Klingon attack.
            if not over:
                shortRange()
                if c == 'Red' and a == 0 and klingonAttack():
                    return True # Game over, we lost.
        elif ans == 's': # Short range scan.
            shortRange()
            if c == 'Red' and a == 0 and klingonAttack():
                return True # Game over, we lost.
        elif ans == 'l': # Long range scan.
            longRange()
        elif ans == 'p': # Phaser control.
            over = phasers()
            if not over:
                over = klingonAttack()
        elif ans == 't': # Photon torpedos.
            over = torpedos()
        elif ans == 'd': # Damage control report.
            damageRpt()
        elif ans == 'e': # Exit
            over = True
        elif ans == 'b': # Library Computer, added by Mike to mimic sttr1.
            computer()
        else:
            tr.print('Commands: c, s, l, p, t, d, e, b.')
    tr.input('\nPress return to exit')

def vec(ex, ey, tx, ty):
    theta = atan2(-(ty-ey), tx-ex)/(pi/4)
    if theta <= 0:
        theta += 8
    theta = theta%8 + 1
    dist = sqrt((tx-ex)**2 + (ty-ey)**2)
    return dist, theta

def warp():
    # Warp drive

    global d,ds,e,k3,q,q1,q2,s1,s2,t

    while True:
        c1 = inputSafe(float, 'Course (1-8.99999)? ')
        w1 = inputSafe(float, 'Warp factor (0-12)? ')
        if w1*c1 == 0:
            return False # Decided against warping.
        elif not (1 <= c1 < 9) or not (0 < w1 <= 12):
            continue # Bad input.
        if d[0] < 0 and w1 > 0.2:
            tr.print('Warp engines are damaged, maximum speed = warp 0.2')
            continue
        break # Break on valid input.
    if k3 > 0: # Klingons in quadrant attack as Enterprise moves.
        if klingonAttack():
            return True # Game over, we lost.
    for i in range(6): # Recover a little from past damage.
        d[i] = min(0, d[i]+1)
    r = random()
    if r < 0.1: # Damage is repaired.
        for i in range(6):
            if d[i] == 0: # Undamaged.
                continue
            d[i] += randint(1, -d[i]) # Repair damage.
            d[i] = min(0, d[i])
            tr.print('*** Truce, %s state of repair improved ***' % ds[i])
    elif r < 0.2: # Equipment is damaged.
        i = randint(0,6)       # Randomly pick equipment to damage.
        d[i] -= randint(1,6) # Damage by random amount.
        d[i] = min(0, d[i])
        tr.print('*** Space storm, %s damaged ***' % ds[i])
    t += 1 # A Solar year passes.
    n = int(8*w1) # Number of sectors to move.
    e -= n # Travel reduces Enterprise energy.
    if e <= 0 or t > t0+t9: # Outta gas or outta time.
        return lose() # Game over.
    q[s1][s2] = 0 # Enterprise leaves sector.
    x,y = s1,s2  # Enterprise current sector is (x,y).
    x1 = -sin((c1 - 1)*pi/4) # Unit vector, course c1 is multiple of 45 deg.
    x2 = cos((c1 - 1)*pi/4)
    for _ in range(n): # Sector at a time, check for objects in the way.
        x += x1 # Update position along unit vector.
        y += x2
        z1 = round(x) # Nearest sector.
        z2 = round(y)
        if not (0 <= z1 <= 7) or not (0 <= z2 <= 7): # Left quadrant.
            # dest = posQuad + posSect + speed*time.
            x = q1 + s1/8 + w1*x1 # Destination quadrant, real.
            y = q2 + s2/8 + w1*x2
            if    x < 0: x = s1 = 0
            elif  x > 7: x = s1 = 7
            else: s1 = int((8*(x - int(x)))%8) # Sector, integer.
            if    y < 0: y = s2 = 0
            elif  y > 7: y = s2 = 7
            else: s2 = int((8*(y - int(y)))%8)
            z1 = int(x) # Quadrant, integer.
            z2 = int(y)
            if (q1,q2) != (z1,z2):
                q1,q2 = z1,z2
                quadrantSetup() # Enables Klingon attack.
            q[s1][s2] = 1 # Enterprise arrives here.
            return False # Game not over.
        if q[z1][z2] != 0:
            tr.print('Enterprise blocked by object at sector %d-%d'
                % (z2+1, z1+1))
            x -= x1
            y -= x2 # Back up along unit vector.
            break
    s1 = int(x)
    s2 = int(y)
    q[s1][s2] = 1 # Enterprise arrives here.
    return False # Game not over.

def welcome():
    global k9,t,t0,t9

    tr.print('trek.bas   27-Jul-73   Aron K. Insinga   Project Delta\n')
    tr.print('\nOrders:   Stardate %d\n' % t)
    tr.print('As commander of the United Starship Enterprise, your mission')
    tr.print('is to rid the galaxy of the deadly Klingon menace.  To do this,')
    tr.print('you must destroy the Klingon invasion force of %d battle' % k9)
    tr.print('cruisers.  You have %d Solar Years to complete your mission.'
        % t9)
    tr.print('(I.e., until Stardate %d.)\n' % (t0+t9))
    tr.print('Give command \'end\' to stop the game early.\n')
    tr.print('Press return to begin.')
    tr.read_key()
    '''
    ans = tr.input('Do you require further instructions? ')
    if ans.lower() == 'y':
        f = open('trek.doc', 'r')
        while True:
            line = f.readline()
            if line == '':
                break
            tr.print(line.rstrip())
        f.close()
    ans = tr.input('Do you want a chart? ')
    if ans.lower() == 'y':
        chartPrint()
    '''

def win():
    global t,t0,k0

    tr.print('\tIt is Stardate %d\n' % t)
    tr.print('The last Klingon battle cruiser in the galaxy has been destroyed.')
    tr.print('The Federation has been saved.')
    tr.print('You have been promoted to Admiral.')
    tr.print('%d Klingons in %d years.  Rating = %d'
        % (k0, t-t0, int(k0/(t-t0)*1000)))
    return True # Game over. :-)

trek()
