'''
Mike Markowski, mike.ab3ap@gmail
May 2021

This class creates an adjacency matrix and reduces it to a fully
interconnected graph.  Nodes with identical DNA matches are collapsed into
one node.

A file dnaCirco.gv in dot language is created for input to Graphviz
that file can be further manipulated if desired.
'''

import numpy as np
import os
import person
import sys

class Circo:

	def build(self, dirFiles, mainPerson, cmLo=-1, cmHi=-1, target=''):
		'''Build an adjacency matrix for a specified Ancestry customer,
		create a Graphviz dot file named dna.gv for the matrix, and call
		Graphviz to create the plot in a file named dna.png.

		Inputs:
		  mainPerson(String) - name of person to make a DNA graph for.  A
		    filename of DNA matches must exist with this name.
		  cmLo (float) - ignore DNA matches who share less than this amount of
		    cM's.
		  cmHi (float) - ignore DNA matches who share more than this amount of
		    cM's.

		Output:
		  ([[...],[...],...]) - adjacency matrix where 1's indicate common
		  matches between people named by row and column, and 0's indicate
		  that there is no match.
		'''

		# Retrieve DNA matches of root person.
		if cmHi == -1:
			cmHi = 7000 # Match everyone if no specified.
		root = person.Person(mainPerson, 7000)
		everyone = root.fileReader(
			dirFiles, mainPerson, cmLo, cmHi, masterList=True)
#		self.printMatrix('Everyone', everyone)
		n = len(everyone)
		if n == 0:
			return # Nothing to do.
		am = []
		for p in everyone:
			am += p.findMatches(dirFiles, everyone, p, cmLo)
		amR = self.reduce(am) # Combine identical rows.
#		self.printPeople('DNA Connections', everyone, amR)
		self.printCirco('dnaCirco.gv', everyone, amR, target)
		return amR

	def matchName(self, everyone, m, row, indent=0):
		'''Convert a matrix row name to a person's Ancestry display name.
		A row name can contain multiple names if multiple rows were collpsed
		into one due to identical common matches.

		Inputs:
		  everyone (list[Persons]) - list all primary DNA matches who also
		    have common matches with other primary matches.
		  m ([[...],[...],...]) - adjacency matrix.
		  row (int) - row of m[] whose names are wanted.
		  indent(int) - number of initial spaces, rarely useful.

		Output:
		  (String) - comma separated list of names from 'everyone'
		    corresponding to hits row of adjacency matrix.
		'''

		names = m[row][0]
		names = names.split(',')
		indexes = list(map(int, names))

		ppl = []
		for i in indexes:
			ppl.append(everyone[i]) # Collect Person instances in node.
		ppl = ppl[0].sortCm(ppl) # Sort with descending cM values.

		s = ''
		for p in ppl: # Convert to string.
			s += indent * ' ' + str(p) + '\n'
		return s[:-1]

	def printCirco(self, gvname, everyone, m, target):
		'''Generate Graphviz dot file.  While there is no return file, two
		things area created: gvname and a png file with the plot.

		Inputs:
		  gvname (String) - name of output Graphviz dot file.
		  everyone (list[Persons]) - list all primary DNA matches who also
		    have common matches with other primary matches.
		  m ([[...],[...],...]) - adjacency matrix.

		Output: none
		'''
		colors = ['red', 'blue', 'green', 'black', 'chartreuse',
			'cyan', 'aqua', 'brown']
		ci = 0
		f = open(gvname, 'w')
		f.write('graph dna {\n')

		f.write('splines=true\n')
		f.write('ratio=expand\n')
		f.write('sep=0.1\n')
		f.write('graph [overlap=false]\n')

		tInd = -1
		for r in range(len(m)): # Each row is a primary match.
			name = self.matchName(everyone, m, r)
			if target != '' and name.find(target) > -1:
				tInd = r # Want to draw thick lines to/from this person.
			f.write('%d [shape=box, label="%s"]\n'
				% (r, self.matchName(everyone, m, r)))
		for r in range(len(m)): # Each row is a primary match.
			primary = self.matchName(everyone, m, r)
			for c in range(r+1, len(m[r])): # Secondary, common matches.
				if m[r][c] == 0:
					continue
				if tInd == r or tInd == c-1:
					f.write('  %d -- %d [color="orange" penwidth=6]\n'
						% (r, c-1))
				else:
					f.write('  %d -- %d [color="%s"]\n'
						% (r, c-1, colors[ci]))
					ci = (ci + 1) % len(colors)
		f.write('}\n')
		f.close()
		pngname = gvname[:gvname.rfind('.')] + '.png'
#		cmd = ('neato -T png -o %s -Gstart=rand %s' % (pngname, gvname))
		cmd = ('circo -T png -o %s %s' % (pngname, gvname))
		os.system(cmd)
		print('%s created.  To regenerate png:' % gvname)
		print('  %s' % cmd)
		print('%s created.' % pngname)

	def reduce(self, am):
		'''After the initial adjacency matrix is made where connection is
		explicit and sometimes redundant, this routine will reduce complexity
		of the matrix by combining rows and columns when possible.

		For example, the matrix:
		 0: ['00' 111111--]
		 1: ['01' 111111--]
		 2: ['02' 111-11--]
		 3: ['03' 11-1----]
		 4: ['04' 111-11--]
		 5: ['05' 111-11--]
		 6: ['06' ------11]
		 7: ['07' ------11]

		can be sorted:
		 0: ['06' ------11]
		 1: ['07' ------11]
		 2: ['03' 11-1----]
		 3: ['02' 111-11--]
		 4: ['04' 111-11--]
		 5: ['05' 111-11--]
		 6: ['00' 111111--]
		 7: ['01' 111111--]

		and the identical rows and corresponding (pre-sorted) columns are
		removed, resulting in a simplified matrix:
		 0: ['00,01' -11-]
		 1: ['02,04,05' 1---]
		 2: ['03' 1---]
		 3: ['06,07' ----]

		Inputs:
		  am ([[...],[...],...]) - adjacency matrix.

		Outut:
		  ([[...],[...],...]) - reduced adjacency matrix.

		'''
		amOrig = am.copy()
#		self.printMatrix('Matrix Raw', am)
		am.sort(key=lambda row: row[1:]) # Sort by row.
#		self.printMatrix('Matrix Sorted', am)
		deleteMe = []
		amNew = []
		row = 0
		while True:
			if row >= len(am):
				break
			rowNew = am[row]
			amNew.append(rowNew)
			while True:
				row += 1
				if row >= len(am):
					break
				rowThis = am[row]
				if rowNew[1:] == rowThis[1:]:
#					print('amNew: %s' % amNew[-1])
					amNew[-1][0] += ',%s' % rowThis[0]
					deleteMe.append(int(rowThis[0]))
				else:
					break
		deleteMe.sort(reverse=True)

		amNew = np.array(amNew)
		for col in deleteMe:
			amNew = np.delete(amNew, col+1, axis=1)
		amNew = amNew.tolist()
		for i in range(len(amNew)):
			row = amNew[i]
			row[1:] = map(int, row[1:])
		amNew.sort()

		for i in range(len(amNew)):
			amNew[i][i+1] = 0
#		self.printMatrix('Matrix Reduced', amNew)

		return amNew

